﻿namespace Microsoft.Samples.PlanMyNight.Web.Controllers
{
    using System;
    using System.Collections.Specialized;
    using System.IO;
    using System.Linq;
    using System.Web;
    using System.Web.Mvc;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.Samples.PlanMyNight.Infrastructure;
    using Microsoft.Samples.PlanMyNight.Infrastructure.Mvc;
    using Microsoft.Samples.PlanMyNight.Web.ViewModels;
    using WindowsLiveId;

    [HandleErrorWithContentType()]
    [OutputCache(NoStore = true, Duration = 0, VaryByParam = "*")]
    public class AccountController : Controller
    {
        private readonly IWindowsLiveLogin windowsLogin;
        private readonly IMembershipService membershipService;
        private readonly IFormsAuthentication formsAuthentication;
        private readonly IReferenceRepository referenceRepository;
        private readonly IActivitiesRepository activitiesRepository;

        public AccountController() :
            this(
                new ServiceFactory().GetMembershipService(),
                new WindowsLiveLogin(true),
                new FormsAuthenticationService(),
                new ServiceFactory().GetReferenceRepositoryInstance(),
                new ServiceFactory().GetActivitiesRepositoryInstance())
        {
        }

        public AccountController(IMembershipService membershipService, IWindowsLiveLogin windowsLogin, IFormsAuthentication formsAuthentication, IReferenceRepository referenceRepository, IActivitiesRepository activitiesRepository)
        {
            this.membershipService = membershipService;
            this.windowsLogin = windowsLogin;
            this.formsAuthentication = formsAuthentication;
            this.referenceRepository = referenceRepository;
            this.activitiesRepository = activitiesRepository;
        }

        public ActionResult LiveId()
        {
            string action = Request.QueryString["action"];
            switch (action)
            {
                case "logout":
                    this.formsAuthentication.SignOut();
                    return Redirect("~/");

                case "clearcookie":
                    this.formsAuthentication.SignOut();
                    string type;
                    byte[] content;
                    this.windowsLogin.GetClearCookieResponse(out type, out content);
                    return new FileStreamResult(new MemoryStream(content), type);

                default:
                    // login
                    NameValueCollection tokenContext;
                    if ((Request.HttpMethod ?? "GET").ToUpperInvariant() == "POST")
                    {
                        tokenContext = Request.Form;
                    }
                    else
                    {
                        tokenContext = new NameValueCollection(Request.QueryString);
                        tokenContext["stoken"] = System.Web.HttpUtility.UrlEncode(tokenContext["stoken"]);
                    }

                    var liveIdUser = this.windowsLogin.ProcessLogin(tokenContext);
                    if (liveIdUser != null)
                    {
                        var returnUrl = liveIdUser.Context;

                        var userId = new Guid(liveIdUser.Id).ToString();
                        if (!this.membershipService.ValidateUser(userId, userId))
                        {
                            this.formsAuthentication.SignIn(userId, false);
                            this.membershipService.CreateUser(userId, userId, string.Empty);
                            var profile = this.membershipService.CreateProfile(userId);
                            profile.FullName = "New User";
                            profile.State = string.Empty;
                            profile.City = string.Empty;
                            profile.PreferredActivityTypeId = 0;
                            this.membershipService.UpdateProfile(profile);
                            if (string.IsNullOrEmpty(returnUrl)) returnUrl = null;
                            return RedirectToAction("Index", new { returnUrl = returnUrl });
                        }
                        else
                        {
                            this.formsAuthentication.SignIn(userId, false);
                            if (string.IsNullOrEmpty(returnUrl)) returnUrl = "~/";
                            return Redirect(returnUrl);
                        }
                    }

                    break;
            }

            return Redirect("~/");
        }

        public ActionResult Login(string returnUrl)
        {
            var redirect = HttpContext.Request.Browser.IsMobileDevice ? this.windowsLogin.GetMobileLoginUrl(returnUrl) : this.windowsLogin.GetLoginUrl(returnUrl);
            return Redirect(redirect);
        }

        [Authorize()]
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Index(string returnUrl)
        {
            var profile = this.membershipService.GetCurrentProfile();
            var model = new ProfileViewModel
            {
                Profile = profile,
                ReturnUrl = returnUrl ?? this.GetReturnUrl()
            };

            this.InjectStatesAndActivityTypes(model);

            return View("Index", model);
        }

        [Authorize()]
        [AcceptVerbs(HttpVerbs.Post)]
        [ValidateAntiForgeryToken()]
        public ActionResult Update(UserProfile profile)
        {
            var returnUrl = Request.Form["returnUrl"];

            if (!ModelState.IsValid)
            {
                // validation error
                return this.IsAjaxCall()
                    ? new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = ModelState }
                    : this.Index(returnUrl);
            }

            this.membershipService.UpdateProfile(profile);

            if (this.IsAjaxCall())
            {
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = new { Update = true, Profile = profile, ReturnUrl = returnUrl } };
            }
            else
            {
                return RedirectToAction("UpdateSuccess", "Account", new { returnUrl = returnUrl });
            }
        }

        public ActionResult UpdateSuccess(string returnUrl)
        {
            var model = new ProfileViewModel
            {
                Profile = this.membershipService.GetCurrentProfile(),
                ReturnUrl = returnUrl
            };

            return View(model);
        }

        private void InjectStatesAndActivityTypes(ProfileViewModel model)
        {
            var profile = model.Profile;

            var types = this.activitiesRepository.RetrieveActivityTypes()
                                                 .Select(o => new SelectListItem { Text = o.Name, Value = o.Id.ToString(), Selected = (profile != null && o.Id == profile.PreferredActivityTypeId) })
                                                 .ToList();
            types.Insert(0, new SelectListItem { Text = "Select...", Value = "0" });

            var states = this.referenceRepository.RetrieveStates()
                                                 .Select(o => new SelectListItem { Text = o.Name, Value = o.Abbreviation, Selected = (profile != null && o.Abbreviation == profile.State) })
                                                 .ToList();
            states.Insert(0, new SelectListItem { Text = "Any state", Value = string.Empty });

            model.PreferredActivityTypes = types;
            model.States = states;
        }
    }
}
